/*!

    \file  i2c2.c

    \brief I2C2 configuration file

*/





#include "gd32f4xx.h"

#include "i2c.h"

#include <stdio.h>



/*!

    \brief      configure the I2C2 GPIO ports

    \param[in]  none

    \param[out] none

    \retval     none

*/

void i2c2_gpio_config(void)

{

		rcu_periph_clock_enable(RCU_GPIOA);

    rcu_periph_clock_enable(RCU_GPIOC);

    

    /* connect PA8 to i2c2_SCL */

    gpio_af_set(GPIOA, GPIO_AF_4, GPIO_PIN_8);

    /* connect PC9 to i2c2_SDA */

    gpio_af_set(GPIOC, GPIO_AF_4, GPIO_PIN_9);



    gpio_mode_set(GPIOA, GPIO_MODE_AF, GPIO_PUPD_PULLUP,GPIO_PIN_8);

    gpio_output_options_set(GPIOA, GPIO_OTYPE_OD, GPIO_OSPEED_50MHZ,GPIO_PIN_8);

    gpio_mode_set(GPIOC, GPIO_MODE_AF, GPIO_PUPD_PULLUP,GPIO_PIN_9);

    gpio_output_options_set(GPIOC, GPIO_OTYPE_OD, GPIO_OSPEED_50MHZ,GPIO_PIN_9);

}



/*!

    \brief      configure the i2c2 interfaces

    \param[in]  none

    \param[out] none

    \retval     none

*/

void i2c2_config(void)

{

    /* enable I2C clock */

    rcu_periph_clock_enable(RCU_I2C2);

    /* configure I2C clock */

    i2c_clock_config(I2C2,I2C2_SPEED,I2C_DTCY_2);

    /* configure I2C address */

    i2c_mode_addr_config(I2C2,I2C_I2CMODE_ENABLE,I2C_ADDFORMAT_7BITS,I2C2_SLAVE_ADDRESS7);

    /* enable I2C2 */

    i2c_enable(I2C2);

    /* enable acknowledge */

    i2c_ack_config(I2C2,I2C_ACK_ENABLE);

}

uint16_t i2c2_address;







/*!

    \brief      initialize slave address used by the I2C  driver

    \param[in]  none

    \param[out] none

    \retval     none

*/

void i2c2_addr_init(uint8_t addr)

{

    i2c2_address = addr;

}



/*!

    \brief      write command to slave

    \param[in]  cmdcommand to control

    \param[out] none

    \retval     none

*/

void i2c2_cmd_write(uint8_t cmd)

{

    /* wait until I2C bus is idle */

    while(i2c_flag_get(I2C2, I2C_FLAG_I2CBSY));



    /* send a start condition to I2C bus */

    i2c_start_on_bus(I2C2);

    

    /* wait until SBSEND bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_SBSEND));

    

    /* send slave address to I2C bus */

		i2c_master_addressing(I2C2, i2c2_address, I2C_TRANSMITTER);

    /* wait until ADDSEND bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_ADDSEND));

    

    /* clear the ADDSEND bit */

    i2c_flag_clear(I2C2,I2C_FLAG_ADDSEND);

    

    /* wait until the transmit data buffer is empty */

    while(SET != i2c_flag_get(I2C2, I2C_FLAG_TBE));

    

    /* send the cmd to I2C bus */

    i2c_data_transmit(I2C2, cmd);

    

    /* wait until BTC bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_BTC));    



    /* send a stop condition to I2C bus */

    i2c_stop_on_bus(I2C2);

    

    /* wait until the stop condition is finished */

    while(I2C_CTL0(I2C2)&0x0200);

}



/*!

    \brief      write one byte to slave

    \param[in]  p_buffer: data

    \param[in]  write_address: slave internal address to write to

    \param[out] none

    \retval     none

*/

void i2c2_byte_write(uint8_t write_address,uint8_t buffer)

{

    /* wait until I2C bus is idle */

    while(i2c_flag_get(I2C2, I2C_FLAG_I2CBSY));



    /* send a start condition to I2C bus */

    i2c_start_on_bus(I2C2);

    

    /* wait until SBSEND bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_SBSEND));

    

    /* send slave address to I2C bus */

    i2c_master_addressing(I2C2, i2c2_address, I2C_TRANSMITTER);

    

    /* wait until ADDSEND bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_ADDSEND));

    

    /* clear the ADDSEND bit */

    i2c_flag_clear(I2C2,I2C_FLAG_ADDSEND);

    

    /* wait until the transmit data buffer is empty */

    while(SET != i2c_flag_get(I2C2, I2C_FLAG_TBE));

    

    /* send the EEPROM's internal address to write to : only one byte address */

    i2c_data_transmit(I2C2, write_address);

    

    /* wait until BTC bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_BTC));

    

    /* send the byte to be written */

    i2c_data_transmit(I2C2, buffer); 

    

    /* wait until BTC bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_BTC));



    /* send a stop condition to I2C bus */

    i2c_stop_on_bus(I2C2);

    

    /* wait until the stop condition is finished */

    while(I2C_CTL0(I2C2)&0x0200);

}



/*!

    \brief      write more than one byte to the slave with a single write cycle

    \param[in]  p_buffer: pointer to the buffer containing the data to be written 

    \param[in]  write_address: slave internal address to write to

    \param[in]  number_of_byte: number of bytes to write to 

    \param[out] none

    \retval     none

*/

void i2c2_write(uint8_t write_address,uint8_t* p_buffer,  uint8_t number_of_byte)

{

    /* wait until I2C bus is idle */

    while(i2c_flag_get(I2C2, I2C_FLAG_I2CBSY));

    

    /* send a start condition to I2C bus */

    i2c_start_on_bus(I2C2);

    

    /* wait until SBSEND bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_SBSEND));

    

    /* send slave address to I2C bus */

    i2c_master_addressing(I2C2, i2c2_address, I2C_TRANSMITTER);

    

    /* wait until ADDSEND bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_ADDSEND));

    

    /* clear the ADDSEND bit */

    i2c_flag_clear(I2C2,I2C_FLAG_ADDSEND);

    

    /* wait until the transmit data buffer is empty */

    while( SET != i2c_flag_get(I2C2, I2C_FLAG_TBE));

    

    /* send the EEPROM's internal address to write to : only one byte address */

    i2c_data_transmit(I2C2, write_address);

    

    /* wait until BTC bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_BTC));

    

    /* while there is data to be written */

    while(number_of_byte--){  

        i2c_data_transmit(I2C2, *p_buffer);

        

        /* point to the next byte to be written */

        p_buffer++; 

        

        /* wait until BTC bit is set */

        while(!i2c_flag_get(I2C2, I2C_FLAG_BTC));

    }

    /* send a stop condition to I2C bus */

    i2c_stop_on_bus(I2C2);

    

    /* wait until the stop condition is finished */

    while(I2C_CTL0(I2C2)&0x0200);

}





/*!

    \brief      read data from the slave

    \param[in]  p_buffer: pointer to the buffer that receives the data read from the slave

    \param[in]  read_address: slave internal address to start reading from

    \param[in]  number_of_byte: number of bytes to reads from the slave

    \param[out] none

    \retval     none

*/

void i2c2_read(uint8_t read_address,uint8_t* p_buffer,  uint16_t number_of_byte)

{  

    /* wait until I2C bus is idle */

    while(i2c_flag_get(I2C2, I2C_FLAG_I2CBSY));



    if(2 == number_of_byte){

        i2c_ackpos_config(I2C2,I2C_ACKPOS_NEXT);

    }

    

    /* send a start condition to I2C bus */

    i2c_start_on_bus(I2C2);

    

    /* wait until SBSEND bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_SBSEND));

    

    /* send slave address to I2C bus */

    i2c_master_addressing(I2C2, i2c2_address, I2C_TRANSMITTER);

    

    /* wait until ADDSEND bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_ADDSEND));

    

    /* clear the ADDSEND bit */

    i2c_flag_clear(I2C2,I2C_FLAG_ADDSEND);

    

    /* wait until the transmit data buffer is empty */

    while(SET != i2c_flag_get( I2C2 , I2C_FLAG_TBE ));



    /* enable I2C2*/

    i2c_enable(I2C2);

    

    /* send the slave internal address to write to */

    i2c_data_transmit(I2C2, read_address);  

    

    /* wait until BTC bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_BTC));

    

    /* send a start condition to I2C bus */

    i2c_start_on_bus(I2C2);

    

    /* wait until SBSEND bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_SBSEND));

    

    /* send slave address to I2C bus */

    i2c_master_addressing(I2C2, i2c2_address, I2C_RECEIVER);



    if(number_of_byte < 3){

        /* disable acknowledge */

        i2c_ack_config(I2C2,I2C_ACK_DISABLE);

    }

    

    /* wait until ADDSEND bit is set */

    while(!i2c_flag_get(I2C2, I2C_FLAG_ADDSEND));

    

    /* clear the ADDSEND bit */

    i2c_flag_clear(I2C2,I2C_FLAG_ADDSEND);

    

    if(1 == number_of_byte){

        /* send a stop condition to I2C bus */

        i2c_stop_on_bus(I2C2);

    }

    

    /* while there is data to be read */

    while(number_of_byte){

        if(3 == number_of_byte){

            /* wait until BTC bit is set */

            while(!i2c_flag_get(I2C2, I2C_FLAG_BTC));



            /* disable acknowledge */

            i2c_ack_config(I2C2,I2C_ACK_DISABLE);

        }

        if(2 == number_of_byte){

            /* wait until BTC bit is set */

            while(!i2c_flag_get(I2C2, I2C_FLAG_BTC));

            

            /* send a stop condition to I2C bus */

            i2c_stop_on_bus(I2C2);

        }

        

        /* wait until the RBNE bit is set and clear it */

        if(i2c_flag_get(I2C2, I2C_FLAG_RBNE)){

            /* read a byte from the EEPROM */

            *p_buffer = i2c_data_receive(I2C2);

            

            /* point to the next location where the byte read will be saved */

            p_buffer++; 

            

            /* decrement the read bytes counter */

            number_of_byte--;

        } 

    }

    

    /* wait until the stop condition is finished */

    while(I2C_CTL0(I2C2)&0x0200);

    

    /* enable acknowledge */

    i2c_ack_config(I2C2,I2C_ACK_ENABLE);



    i2c_ackpos_config(I2C2,I2C_ACKPOS_CURRENT);

}







